// This MFC Samples source code demonstrates using MFC Microsoft Office Fluent User Interface 
// (the "Fluent UI") and is provided only as referential material to supplement the 
// Microsoft Foundation Classes Reference and related electronic documentation 
// included with the MFC C++ library software.  
// License terms to copy, use or distribute the Fluent UI are available separately.  
// To learn more about our Fluent UI licensing program, please visit 
// http://msdn.microsoft.com/officeui.
//
// Copyright (C) Microsoft Corporation
// All rights reserved.

// bbDemoView.cpp : implementation of the CbbDemoView class
//
#include "stdafx.h"
#pragma warning(disable:4244) // float to int, vice versa conversions
#pragma warning(disable:4482) // enum warning
#include "bbDemo.h"

#include "bbDemoDoc.h"
#include "bbDemoView.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#endif


// CbbDemoView

IMPLEMENT_DYNCREATE(CbbDemoView, CView)

BEGIN_MESSAGE_MAP(CbbDemoView, CView)
	ON_WM_ERASEBKGND()
	ON_WM_SIZE()
	ON_WM_CREATE()
	ON_WM_LBUTTONDOWN()
	ON_WM_RBUTTONDOWN()
	ON_WM_MOUSEWHEEL()
END_MESSAGE_MAP()

// CbbDemoView construction/destruction

CbbDemoView::CbbDemoView()
{
	// TODO: add construction code here
}

//
// Deconstructor
//
CbbDemoView::~CbbDemoView()
{
	graticule.clear();
	signal.clear();

	wglMakeCurrent(m_cDC->GetSafeHdc(), m_hRC);

	// Delete VBO's
	glDeleteBuffers(1, &signalVBO);
	glDeleteBuffers(1, &gratVBO);
	// Delete Fonts
	glDeleteLists(fontHandle, 96);

	wglMakeCurrent(NULL, NULL);
}

// 
// glInit
//
void CbbDemoView::glInit()
{
	wglMakeCurrent( m_cDC->GetSafeHdc(), m_hRC ); 

	glewInit();
	glShadeModel(GL_SMOOTH);
	glClearDepth(1.0f);
	glEnable(GL_DEPTH_TEST);
	glDepthFunc(GL_LEQUAL);

	glClearColor(0.133f, 0.156f, 0.164f, 1.0f); // Slate background color
	glEnableClientState(GL_VERTEX_ARRAY);

	glHint(GL_LINE_SMOOTH_HINT, GL_NICEST);

	glGenBuffers(1, &gratVBO);
	glGenBuffers(1, &signalVBO);
	createGraticule();

	// Create and use our font
	fontHandle = glGenLists(96);
	font = getFont(22, FW_NORMAL, "Arial");
	SelectObject(m_cDC->GetSafeHdc(), font);
	wglUseFontBitmaps(m_cDC->GetSafeHdc(), 32, 96, fontHandle);

	wglMakeCurrent(NULL, NULL);
}

//
// Added OwnDC for gl functionality
//
BOOL CbbDemoView::PreCreateWindow(CREATESTRUCT& cs)
{
	// TODO: Modify the Window class or styles here by modifying
	//  the CREATESTRUCT cs
	cs.style |= ( WS_CLIPCHILDREN | WS_CLIPSIBLINGS | CS_OWNDC );
	return CView::PreCreateWindow(cs);
}

//
// Override to setup our gl context
//
int CbbDemoView::OnCreate(LPCREATESTRUCT lpCreateStruct)
{
	if (CView::OnCreate(lpCreateStruct) == -1)
		return -1;

	PIXELFORMATDESCRIPTOR pfd; 
    int pixelFormat; 

	m_cDC = new CClientDC( this );

	// Setup our pfd
	pfd.nSize =             sizeof(PIXELFORMATDESCRIPTOR); 
	pfd.nVersion =          1; 
	pfd.dwFlags =           PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pfd.iPixelType =        PFD_TYPE_RGBA; 
	pfd.cColorBits =        24; 

	pixelFormat = ChoosePixelFormat(m_cDC->GetSafeHdc(), &pfd); 
	DescribePixelFormat(
		m_cDC->GetSafeHdc(), 
		pixelFormat, 				
		sizeof(PIXELFORMATDESCRIPTOR), 					
		&pfd); 

	if(SetPixelFormat(m_cDC->GetSafeHdc(), pixelFormat, &pfd) == FALSE) {
		// This should throw a messageBox	
		exit(1); 
	}

	m_hRC = wglCreateContext(m_cDC->GetSafeHdc()); 
	glInit();     
	return 0;
}

//
// OnDraw
//
void CbbDemoView::OnDraw(CDC* /*pDC*/)
{
	CbbDemoDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (!pDoc)
		return;

	// Get some dimensions
	gratX = viewWidth - margin*2;
	gratY = viewHeight - margin*2;
	wglMakeCurrent(m_cDC->GetSafeHdc(), m_hRC);

	// Setup matrices
	glViewport(0, 0, viewWidth, viewHeight);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(0, viewWidth, 0, viewHeight, -1, 1);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
 
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	// Draw Text, prior to scaling matrices
	glColor3ub(86, 156, 214);
	glColor3f(1.0f, 1.0f, 1.0f);
	rastText("BB60A Demo", CENTER, margin+0.5f*gratX, margin+gratY+27);
	rastText("Use Your Left and Right Mouse Buttons", CENTER, margin+0.5f*gratX, margin-40);
	rastText("Start 88 MHz", LEFT, margin, margin-20);
	rastText("Attach a FM antenna", LEFT, margin, margin-40);
	rastText("Span 20 MHz", CENTER, margin+0.5f*gratX, margin-20);
	rastText("Stop 108 MHz", RIGHT, margin+gratX, margin-20);
	rastText("Ref 0 dBm", LEFT, margin, margin+gratY+7);
	rastText("Div 10 dB", CENTER, margin+0.5f*gratX, margin+gratY+7);
	rastText("RBW 19.7 kHz", RIGHT, margin+gratX, margin+gratY+7);

	// Text showing the current mode
	if(pDoc->mode == pDoc->IOMode::IDLE) {
		glColor3f(1.0f, 0.0f, 0.0f);
		rastText("Idle", LEFT, margin+5, margin+gratY-20);
	} else if(pDoc->mode == pDoc->IOMode::FROMDEV) {
		glColor3f(1.0f, 1.0f, 1.0f);
		rastText("Live From Device", LEFT, margin+5, margin+gratY-20);
	} else if(pDoc->mode == pDoc->IOMode::TODISK) {
		glColor3f(1.0f, 0.0f, 0.0f);
		rastText("Live Recording", LEFT, margin+5, margin+gratY-20);
	} else if(pDoc->mode == pDoc->IOMode::FROMDISK) {
		glColor3f(0.0f, 1.0f, 0.0f);
		rastText("Playback From File", LEFT, margin+5, margin+gratY-20);
	}

	// Prep modelview for grat and signal
	glMatrixMode(GL_MODELVIEW);
	glTranslatef((float)margin, (float)margin, 0.0f);
	glScalef((float)gratX, (float)gratY, 1.0f);

	// Draw graticule
	glColor3f(1.0f, 1.0f, 1.0f);
	glLineStipple(1, 0x8888);
	glEnable(GL_LINE_STIPPLE);
	glBindBuffer(GL_ARRAY_BUFFER , gratVBO);
	glVertexPointer(2, GL_FLOAT, 0, (GLvoid*)0);
	glDrawArrays(GL_LINES, 0, graticule.size()/2);	
	glDisable(GL_LINE_STIPPLE);

	// Draw signal
	glColor3f(1.0f, 1.0f, 0.0f);
	normalizeSignal();
	drawSweep();

	// Draw our two channels
	drawChannel(Channel::LEFT, pDoc->leftFreq);
	drawChannel(Channel::RIGHT, pDoc->rightFreq);

	glFinish();

	SwapBuffers(m_cDC->GetSafeHdc());

	wglMakeCurrent(NULL, NULL);
}

//
// Draws our signal, simply a line strip
//
void CbbDemoView::drawSweep()
{
	if(signal.size() < 1) return;

	glLineWidth(2.0);
	glBindBuffer(GL_ARRAY_BUFFER, signalVBO);
	glBufferData(GL_ARRAY_BUFFER, signal.size()*sizeof(float),
		&signal[0], GL_DYNAMIC_DRAW);
	glVertexPointer(2, GL_FLOAT, 0, (GLvoid*)0);

	glDrawArrays(GL_LINE_STRIP, 0, signal.size()/2);
	glLineWidth(1.0);
}

//
// Draws a thin channel box, with text
//
void CbbDemoView::drawChannel(Channel c, float f)
{
	CString station, chan;
	float x = (20.0e6-(108e6-f))/20.0e6;
	int off = c == Channel::RIGHT ? -40 : 0;

	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
	glColor4f(0.5f, 0.5f, 0.5f, 0.4f);
	glBegin(GL_QUADS);
	glVertex2f(x-0.005f, 0.0f);
	glVertex2f(x+0.005f, 0.0f);
	glVertex2f(x+0.005f, 1.0f);
	glVertex2f(x-0.005f, 1.0f);
	glEnd();
	glDisable(GL_BLEND);

	glColor4f(1,1,1,1);
	glPushMatrix();
	glLoadIdentity();
	station.Format("%f", f/1.0e6);
	chan.Format("%s", c == Channel::LEFT ? "Left" : "Right");
	rastText(chan, CENTER, margin+x*gratX, margin+0.9f*gratY+20+off);
	rastText(station, CENTER, margin+x*gratX, margin+0.9f*gratY+off);
	glPopMatrix();
}

//
// Turns a freq domain signal to 0.0-1.0 normalized coords
//
void CbbDemoView::normalizeSignal()
{
	CbbDemoDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (!pDoc)
		return;

	// Sweep size check, draw before everything setup
	if(pDoc->view == NULL) {
		signal.clear();	
		return;
	}

	signal.clear();
	signal.reserve(4096);

	float ref = 0.0f;
	float bot = -100.0f;

	float xStep = 1.0f/FFTDIV4;
	float xScale = 1.0f/gratX;
	float yScale = 1.0f/100.0f;
	
	for(unsigned i = 0; i < FFTDIV4; i++)
	{
		float y = yScale*(pDoc->view[i]-bot);
		y = min(y, 1.0f);
		y = max(y, 0.0f);

		signal.push_back(i*xStep);
		signal.push_back(y);
	}
}

//
// One time up front graticule creation
// 
void CbbDemoView::createGraticule()
{
	float gratDiv = 0.1f;
	
	graticule.reserve(88);
	for(int i = 0; i <= 10; i++) {
		graticule.push_back(0.0f);
		graticule.push_back(gratDiv*i);
		
		graticule.push_back(1.0f);
		graticule.push_back(gratDiv*i);
	}
	
	for(int i = 0; i <= 10; i++) {
		graticule.push_back(gratDiv*i);
		graticule.push_back(0.0f);
		
		graticule.push_back(gratDiv*i);
		graticule.push_back(1.0f);
	}

	glBindBuffer( GL_ARRAY_BUFFER , gratVBO );
	glBufferData( GL_ARRAY_BUFFER, graticule.size()*sizeof(float), 
				  &graticule[0], GL_STATIC_DRAW );
}

//
// Creates a bitmap font
//
HFONT CbbDemoView::getFont( int h, int w, char * name ) 
{
	return CreateFontA(
		h,                                // Height
		0,                                // Width
		0,                                // Angle of escaptement
		0,                                // Orientation angle
		w,                                // Font Width
		FALSE,                            // Italic
		FALSE,                            // Underline
		FALSE,                            // Striked-out
		ANSI_CHARSET,                     // Charset
		OUT_TT_PRECIS,                    // Precision
		CLIP_DEFAULT_PRECIS,              // Clip precision
		ANTIALIASED_QUALITY,              // Output quality
		FF_DONTCARE | DEFAULT_PITCH,      // Family and Pitch
		name );                           // Font name
}

//
// Main rast text function
//
void CbbDemoView::rastText(const CString& s, 
						   JUSTIFICATION just, 
						   int x, 
						   int y)
{
	int xAdjusted = x;  // x value after adjusting for justification
	CRect r;
	DrawText(m_cDC->GetSafeHdc(), s, -1, r, DT_CALCRECT);

	switch( just ) {
		case LEFT:
			xAdjusted = x;
			break;
		case RIGHT:
			xAdjusted = x - r.Width();
			break;
		case CENTER:
			xAdjusted = x - r.Width()/2;
			break;
		default: // Don't get here
			xAdjusted = x;
			break;
	}
	glRasterPos2i(xAdjusted, y);
	myPrint(s);
}

//
// Rasts a bitmap str at the current glRasterPos
//
void CbbDemoView::myPrint(const char* s)
{
	if(s == NULL) return;

	glPushAttrib(GL_LIST_BIT);
	glListBase(fontHandle - 32);
	glCallLists(strlen(s), GL_UNSIGNED_BYTE, s);
	glPopAttrib();
}

void CbbDemoView::OnRButtonUp(UINT nFlags, CPoint point)
{
	ClientToScreen(&point);
	OnContextMenu(this, point);
}

void CbbDemoView::OnContextMenu(CWnd* pWnd, CPoint point)
{
	theApp.GetContextMenuManager()->ShowPopupMenu(IDR_POPUP_EDIT, point.x, point.y, this, TRUE);
}


// CbbDemoView diagnostics

#ifdef _DEBUG
void CbbDemoView::AssertValid() const
{
	CView::AssertValid();
}

void CbbDemoView::Dump(CDumpContext& dc) const
{
	CView::Dump(dc);
}

CbbDemoDoc* CbbDemoView::GetDocument() const // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CbbDemoDoc)));
	return (CbbDemoDoc*)m_pDocument;
}
#endif //_DEBUG


// CbbDemoView message handlers

BOOL CbbDemoView::OnEraseBkgnd(CDC* pDC)
{
	return true;
}

//
// Resize, adjust dimensions
// OnDraw gets called automatically
//
void CbbDemoView::OnSize(UINT nType, int cx, int cy)
{
	CView::OnSize(nType, cx, cy);
	viewWidth = cx;
	viewHeight = cy - 20;
}

//
// Change left channel
//
void CbbDemoView::OnLButtonDown(UINT nFlags, CPoint point)
{
	CbbDemoDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (!pDoc)
		return;

	int x = point.x - margin;
	float d = (float)gratX/100.0f;
	x = (int)((float)x/d);

	pDoc->setStation(Channel::LEFT, 88.1e6 + x*200.e3);
	CView::OnLButtonDown(nFlags, point);

	Invalidate(0);
}

//
// Change right channel
//
void CbbDemoView::OnRButtonDown(UINT nFlags, CPoint point)
{
	CbbDemoDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (!pDoc)
		return;

	int x = point.x - margin;
	float d = (float)gratX/100.0f;
	x = (int)((float)x/d);

	pDoc->setStation(Channel::RIGHT, 88.1e6 + x*200.e3);
	CView::OnRButtonDown(nFlags, point);

	Invalidate(0);
}

//
// Mouse wheel to change station
//
BOOL CbbDemoView::OnMouseWheel(UINT nFlags, short zDelta, CPoint pt)
{
	CbbDemoDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (!pDoc)
		return false;

	pDoc->slideStation(zDelta);
	return CView::OnMouseWheel(nFlags, zDelta, pt);
}
